/*
 * Decompiled with CFR 0.152.
 */
package gaiasky.desktop.util;

import com.badlogic.gdx.Gdx;
import com.badlogic.gdx.backends.lwjgl3.Lwjgl3Files;
import com.badlogic.gdx.files.FileHandle;
import com.beust.jcommander.JCommander;
import com.beust.jcommander.Parameter;
import gaiasky.gui.main.ConsoleLogger;
import gaiasky.scene.record.BillboardDataset;
import gaiasky.util.Logger;
import gaiasky.util.SettingsManager;
import gaiasky.util.i18n.I18n;
import gaiasky.util.math.MathUtilsDouble;
import gaiasky.util.math.StdRandom;
import gaiasky.util.math.Vector3D;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.zip.GZIPOutputStream;
import net.jafama.FastMath;

public class GalaxyGenerator {
    private static final Logger.Log logger = Logger.getLogger(GalaxyGenerator.class);
    private static final String separator = " ";
    private static final boolean writeFile = true;
    private static final int N = 3000;
    private static final int Narms = 8;
    private static final boolean bar = true;
    private static final double barLength = 0.8;
    private static final double radius = 2.5;
    private static final double armWidthRatio = 0.04;
    private static final double armHeightRatio = 0.02;
    private static final double maxRotation = 100.0;
    private static final boolean radialDensity = true;
    private static CLIArgs cliArgs;

    public static void main(String[] args) {
        try {
            Gdx.files = new Lwjgl3Files();
            cliArgs = new CLIArgs();
            JCommander jc = JCommander.newBuilder().addObject((Object)cliArgs).build();
            jc.setProgramName("galaxy-generator");
            try {
                jc.parse(args);
                if (GalaxyGenerator.cliArgs.help) {
                    jc.usage();
                    return;
                }
            }
            catch (Exception e) {
                logger.error("galaxy-generator: bad program arguments\n\n");
                jc.usage();
                return;
            }
            SettingsManager.initialize(new FileInputStream("assets/conf/config.yaml"), new FileInputStream("assets/dummyversion"));
            I18n.initialize(new FileHandle("assets/i18n/gsbundle"), new FileHandle("assets/i18n/objects"));
            new ConsoleLogger();
            StdRandom.setSeed(System.currentTimeMillis());
            List<double[]> gal = switch (GalaxyGenerator.cliArgs.galaxyType.ordinal()) {
                default -> throw new MatchException(null, null);
                case 0 -> GalaxyGenerator.generateGalaxySpiral();
                case 1 -> GalaxyGenerator.generateMilkyWay();
                case 2 -> GalaxyGenerator.generateUniform();
                case 3 -> GalaxyGenerator.generateBulge();
            };
            GalaxyGenerator.writeToDisk(gal, GalaxyGenerator.cliArgs.outputDir);
        }
        catch (Exception e) {
            logger.error(e);
        }
    }

    private static double generateNewSize() {
        return switch (GalaxyGenerator.cliArgs.particleType) {
            case BillboardDataset.ParticleType.STAR, BillboardDataset.ParticleType.DUST -> FastMath.abs((double)(StdRandom.uniform() * 20.0 + StdRandom.uniform() * 3.0));
            case BillboardDataset.ParticleType.BULGE -> FastMath.abs((double)(StdRandom.uniform() * 40.0 + StdRandom.uniform() * 6.0));
            case BillboardDataset.ParticleType.HII -> FastMath.abs((double)(StdRandom.uniform() * 70.0 + StdRandom.uniform() * 30.0));
            case BillboardDataset.ParticleType.GAS -> FastMath.abs((double)(StdRandom.uniform() * 100.0 + StdRandom.uniform() * 50.0));
            default -> 1.0;
        };
    }

    private static double[] cl(double r, double g, double b) {
        return new double[]{MathUtilsDouble.clamp(r, 0.0, 1.0), MathUtilsDouble.clamp(g, 0.0, 1.0), MathUtilsDouble.clamp(b, 0.0, 1.0)};
    }

    private static double[] generateNewColor() {
        double r = StdRandom.gaussian();
        switch (GalaxyGenerator.cliArgs.particleType) {
            case STAR: {
                r *= 0.15;
                if (StdRandom.uniform(2) == 0) {
                    return GalaxyGenerator.cl(0.95 - r, 0.8 - r, 0.6);
                }
                return GalaxyGenerator.cl(0.95, 0.8 - r, 0.6 - r);
            }
            case BULGE: {
                return new double[]{0.9, 0.9, 0.8};
            }
            case HII: {
                return new double[]{0.78, 0.31, 0.55};
            }
            case GAS: {
                return GalaxyGenerator.cl(0.068 + (r *= 0.1), 0.06 + r, 0.2 + r * 1.3);
            }
        }
        return null;
    }

    private static void addMWParticle(double x, double y, double z, Vector3D aux, List<double[]> particles) {
        aux.set(x, y, z);
        double size = GalaxyGenerator.generateNewSize();
        double[] color = GalaxyGenerator.generateNewColor();
        if (color == null || color.length < 3) {
            particles.add(new double[]{x, y, z, size});
        } else {
            particles.add(new double[]{x, y, z, size, color[0], color[1], color[2]});
        }
    }

    private static List<double[]> generateUniform() {
        return GalaxyGenerator.generateUniformBlob(0.14285714285714285, 0.14285714285714285, 0.025);
    }

    private static List<double[]> generateBulge() {
        return GalaxyGenerator.generateUniformBlob(0.15, 0.15, 0.05);
    }

    private static List<double[]> generateUniformBlob(double xExtent, double yExtent, double zExtent) {
        Vector3D aux = new Vector3D();
        ArrayList<double[]> particles = new ArrayList<double[]>(3000);
        for (int i = 0; i < 3000; ++i) {
            double x = 0.5 * StdRandom.gaussian(0.0, 10.0) * xExtent;
            double y = 0.5 * StdRandom.gaussian(0.0, 10.0) * yExtent;
            double z = StdRandom.gaussian() * zExtent;
            GalaxyGenerator.addMWParticle(x, y, z, aux, particles);
        }
        return particles;
    }

    private static List<double[]> generateMilkyWay() {
        double z;
        double y;
        double x;
        int i;
        Vector3D aux = new Vector3D();
        ArrayList<double[]> particles = new ArrayList<double[]>(3000);
        int Nbar = 300;
        int Nbulge = 500;
        int Nrest = 3500;
        for (i = 0; i < Nbar; ++i) {
            x = StdRandom.gaussian(0.0, 0.18);
            y = StdRandom.gaussian(0.0, 0.03);
            z = StdRandom.gaussian(0.0, 0.041666666666666664);
            GalaxyGenerator.addMWParticle(x, y, z, aux, particles);
        }
        for (i = 0; i < Nbulge; ++i) {
            x = StdRandom.gaussian(0.0, 0.18);
            y = StdRandom.gaussian(0.0, 0.18);
            z = StdRandom.gaussian(0.0, 0.041666666666666664);
            GalaxyGenerator.addMWParticle(x, y, z, aux, particles);
        }
        for (i = 0; i < Nrest; ++i) {
            x = StdRandom.gaussian();
            y = StdRandom.gaussian();
            z = StdRandom.gaussian(0.0, 0.03333333333333333);
            GalaxyGenerator.addMWParticle(x, y, z, aux, particles);
        }
        for (double[] particle : particles) {
            aux.set(particle[0], particle[1], particle[2]);
            aux.rotate(-45.0, 0.0, 0.0, 1.0);
            particle[0] = aux.x;
            particle[1] = aux.y;
            particle[2] = aux.z;
        }
        return particles;
    }

    private static List<double[]> generateGalaxySpiral() throws RuntimeException {
        double y;
        double x;
        double totalLength = 20.8;
        double armOverTotal = 2.5 / totalLength;
        double barOverTotal = 0.8 / totalLength;
        long NperArm = FastMath.round((double)(3000.0 * armOverTotal));
        long Nbar = FastMath.round((double)(3000.0 * barOverTotal));
        double armWidth = 0.1;
        double armHeight = 0.05;
        ArrayList<double[]> particles = new ArrayList<double[]>(3000);
        double stepAngle = 60.0 / FastMath.max((double)1.0, (double)3.0);
        double angle = 10.0;
        Vector3D rotAxis = new Vector3D(0.0, 1.0, 0.0);
        for (long j = 0L; j < Nbar; ++j) {
            double z = StdRandom.uniform() * 0.8 - 0.4;
            x = StdRandom.gaussian() * armWidth;
            y = StdRandom.gaussian() * armHeight;
            particles.add(new double[]{x, y, z, FastMath.abs((double)StdRandom.gaussian())});
        }
        for (int i = 0; i < 8; ++i) {
            logger.info("Generating arm " + (i + 1));
            double zPlus = 0.4 * ((double)i < 4.0 ? 1.0 : -1.0);
            angle = (double)i == 4.0 ? 190.0 : angle;
            int j = 0;
            while ((long)j < NperArm) {
                double z = FastMath.abs((double)StdRandom.gaussian()) * 2.5;
                x = StdRandom.gaussian() * armWidth;
                y = StdRandom.gaussian() * armHeight;
                Vector3D particle = new Vector3D(x, y, z);
                particle.rotate(rotAxis, angle);
                particle.rotate(rotAxis, 100.0 * particle.len() / 2.5);
                particle.add(0.0, 0.0, zPlus);
                particles.add(new double[]{particle.x, particle.y, particle.z, FastMath.abs((double)StdRandom.gaussian())});
                ++j;
            }
            angle += stepAngle;
        }
        return particles;
    }

    private static void writeToDisk(List<double[]> gal, String dir) throws IOException {
        gal.sort(Comparator.comparingDouble(f -> f[0]));
        String filePath = dir + "galaxy_";
        filePath = GalaxyGenerator.cliArgs.galaxyType == GalaxyType.spiral ? filePath + "bar0.8_8arms_3000particles_2.5radius_0.04ratio_100.0deg.dat.gz" : filePath + "3000particles.dat.gz";
        FileHandle fh = new FileHandle(filePath);
        File f2 = fh.file();
        if (fh.exists() && f2.isFile()) {
            fh.delete();
        }
        if (fh.isDirectory()) {
            throw new RuntimeException("File is directory: " + filePath);
        }
        if (f2.createNewFile()) {
            BufferedWriter bw = GalaxyGenerator.getBufferedWriter(gal, filePath);
            bw.close();
            logger.info(I18n.msg("notif.written", gal.size(), filePath));
        } else {
            logger.error("Error creating file: " + f2.getAbsolutePath());
        }
    }

    private static BufferedWriter getBufferedWriter(List<double[]> gal, String filePath) throws IOException {
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(new GZIPOutputStream(new FileOutputStream(filePath))));
        bw.write("X Y Z size r g b");
        bw.newLine();
        for (double[] star : gal) {
            StringBuilder sb = new StringBuilder();
            sb.append(star[0]);
            for (int j = 1; j < star.length; ++j) {
                sb.append(separator);
                sb.append(star[j]);
            }
            bw.write(sb.toString());
            bw.newLine();
        }
        return bw;
    }

    private static class CLIArgs {
        @Parameter(names={"-h", "--help"}, description="Show program options and usage information.", help=true, order=0)
        private boolean help = false;
        @Parameter(names={"--galaxytype"}, required=true, description="The galaxy type to use.", order=1)
        private GalaxyType galaxyType = GalaxyType.uniform;
        @Parameter(names={"--particletype"}, required=true, description="The particle type to use.", order=2)
        private BillboardDataset.ParticleType particleType = BillboardDataset.ParticleType.GAS;
        @Parameter(names={"-o", "--ouptut"}, description="The output directory.", required=true, order=2)
        private String outputDir = "/home/tsagrista/.local/share/gaiasky/data/galaxy/";
        @Parameter(names={"-s", "--skip-welcome"}, description="Skip the welcome screen if possible (base-data package must be present).", order=2)
        private boolean skipWelcome = false;

        private CLIArgs() {
        }
    }

    static enum GalaxyType {
        spiral,
        milkyway,
        uniform,
        bulge;

    }
}

