/*
 * Decompiled with CFR 0.152.
 */
package gaiasky.util.gdx.loader;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

public final class PortableFloatMap {
    public int width;
    public int height;
    public Mode mode;
    public float[] pixels;
    public boolean bigEndian;

    public PortableFloatMap() {
        this.bigEndian = true;
    }

    public PortableFloatMap(File file) throws IOException {
        Objects.requireNonNull(file);
        try (BufferedInputStream in = new BufferedInputStream(new FileInputStream(file));){
            this.read(in);
        }
    }

    public PortableFloatMap(InputStream in) throws IOException {
        Objects.requireNonNull(in);
        this.read(in);
    }

    private static String readLine(InputStream in) throws IOException {
        byte[] buf = new byte[100];
        for (int i = 0; i < buf.length; ++i) {
            int b = in.read();
            if (b == 10 || b == -1) {
                return new String(buf, 0, i, StandardCharsets.US_ASCII);
            }
            buf[i] = (byte)b;
        }
        throw new IllegalArgumentException("Line too long");
    }

    private void read(InputStream in) throws IOException {
        String format = PortableFloatMap.readLine(in);
        if (format.equals("PF")) {
            this.mode = Mode.COLOR;
        } else if (format.equals("Pf")) {
            this.mode = Mode.GRAYSCALE;
        } else {
            throw new IllegalArgumentException("Unrecognized format: " + format);
        }
        String[] tokens = PortableFloatMap.readLine(in).split(" ", 2);
        if (tokens.length != 2) {
            throw new IllegalArgumentException("Invalid dimensions");
        }
        try {
            this.width = Integer.parseInt(tokens[0]);
            this.height = Integer.parseInt(tokens[1]);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException("Invalid dimensions");
        }
        if (this.width <= 0 || this.height <= 0) {
            throw new IllegalArgumentException("Width and height must be positive");
        }
        double temp = Double.parseDouble(PortableFloatMap.readLine(in));
        if (temp >= 0.0) {
            this.bigEndian = true;
        } else if (temp < 0.0) {
            this.bigEndian = false;
        } else {
            throw new IllegalArgumentException("Unrecognized format: " + format);
        }
        DataInputStream din = new DataInputStream(in);
        this.pixels = new float[this.calcPixelArrayLength()];
        if (this.bigEndian) {
            for (int i = 0; i < this.pixels.length; ++i) {
                this.pixels[i] = din.readFloat();
            }
        } else {
            for (int i = 0; i < this.pixels.length; ++i) {
                this.pixels[i] = Float.intBitsToFloat(Integer.reverseBytes(din.readInt()));
            }
        }
    }

    public void write(File file) throws IOException {
        Objects.requireNonNull(file);
        this.checkData();
        try (BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file));){
            this.write(out);
        }
    }

    public void write(OutputStream out) throws IOException {
        Objects.requireNonNull(out);
        this.checkData();
        PrintWriter pout = new PrintWriter(new OutputStreamWriter(out, StandardCharsets.US_ASCII));
        switch (this.mode.ordinal()) {
            case 0: {
                pout.print("PF\n");
                break;
            }
            case 1: {
                pout.print("Pf\n");
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        pout.print(this.width + " " + this.height + "\n");
        pout.print((this.bigEndian ? "1.0" : "-1.0") + "\n");
        pout.flush();
        DataOutputStream dataOutput = new DataOutputStream(out);
        if (this.bigEndian) {
            for (float x : this.pixels) {
                dataOutput.writeFloat(x);
            }
        } else {
            for (float x : this.pixels) {
                dataOutput.writeInt(Integer.reverseBytes(Float.floatToIntBits(x)));
            }
        }
    }

    private void checkData() {
        if (this.width <= 0) {
            throw new IllegalStateException("Width must be positive");
        }
        if (this.height <= 0) {
            throw new IllegalStateException("Height must be positive");
        }
        Objects.requireNonNull(this.mode, "Mode not set");
        Objects.requireNonNull(this.pixels, "Pixel array not set");
        if (this.pixels.length != this.calcPixelArrayLength()) {
            throw new IllegalStateException("Pixel array length does not match width and height");
        }
    }

    private int calcPixelArrayLength() {
        int channels = switch (this.mode.ordinal()) {
            case 0 -> 3;
            case 1 -> 1;
            default -> throw new AssertionError();
        };
        if (Integer.MAX_VALUE / this.width / this.height / channels == 0) {
            throw new IllegalArgumentException("Dimensions are too large to make a pixel array");
        }
        return this.width * this.height * channels;
    }

    public static enum Mode {
        COLOR,
        GRAYSCALE;

    }
}

