/*
 * Decompiled with CFR 0.152.
 */
package gaiasky.util.math;

import gaiasky.util.math.QuadrupleParser;
import java.util.Arrays;

public class Quadruple
extends Number
implements Comparable<Quadruple> {
    private static final int HASH_CODE_OF_NAN = -441827835;
    public static final int EXPONENT_OF_ONE = Integer.MAX_VALUE;
    public static final int EXPONENT_BIAS = Integer.MAX_VALUE;
    public static final long EXPONENT_OF_MAX_VALUE = 0xFFFFFFFEL;
    public static final int EXPONENT_OF_INFINITY = -1;
    private boolean negative;
    private int exponent;
    private long mantHi;
    private long mantLo;
    private static final long HIGH_BIT = Long.MIN_VALUE;
    private static final long BIT_63 = Long.MIN_VALUE;
    private static final long LOWER_32_BITS = 0xFFFFFFFFL;
    private static final long HIGHER_32_BITS = -4294967296L;
    private static final long DOUBLE_SIGN_MASK = Long.MIN_VALUE;
    private static final long DOUBLE_EXP_MASK = 0x7FF0000000000000L;
    private static final long DOUBLE_MANT_MASK = 0xFFFFFFFFFFFFFL;
    private static final int EXP_0D = 1023;
    private static final long HALF_DOUBLES_LSB = 2048L;
    private static final long DOUBLE_IMPLIED_MSB = 0x10000000000000L;
    private static final Quadruple ONE = new Quadruple().assign(1L);
    private static final ThreadLocal<Buffers> buffers = ThreadLocal.withInitial(Buffers::new);
    private static final int[] SQUARE_BYTES = new int[]{0, 513, 1028, 1545, 2064, 2585, 3108, 3633, 4160, 4689, 5220, 5753, 6288, 6825, 7364, 7905, 8448, 8993, 9540, 10089, 10640, 11193, 11748, 12305, 12864, 13425, 13988, 14553, 15120, 15689, 16260, 16833, 17408, 17985, 18564, 19145, 19728, 20313, 20900, 21489, 22080, 22673, 23268, 23865, 24464, 25065, 25668, 26273, 26880, 27489, 28100, 28713, 29328, 29945, 30564, 31185, 31808, 32433, 33060, 33689, 34320, 34953, 35588, 36225, 36864, 37505, 38148, 38793, 39440, 40089, 40740, 41393, 42048, 42705, 43364, 44025, 44688, 45353, 46020, 46689, 47360, 48033, 48708, 49385, 50064, 50745, 51428, 52113, 52800, 53489, 54180, 54873, 55568, 56265, 56964, 57665, 58368, 59073, 59780, 60489, 61200, 61913, 62628, 63345, 64064, 64785, 65508};
    private static final int[] ROOT_BYTES = new int[]{0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106};
    private static final int WORD_LENGTH = 64;
    private static final int DIGIT_LENGTH = 8;
    private static final int MAX_BITS_FOR_SQRT = 160;

    public Quadruple() {
    }

    public Quadruple(Quadruple qValue) {
        this.assign(qValue);
    }

    public Quadruple(double dValue) {
        this.assign(dValue);
    }

    public Quadruple(long lValue) {
        this.assign(lValue);
    }

    public Quadruple(String strValue) {
        this.assign(strValue);
    }

    public Quadruple(boolean negative, int exponent, long mantHi, long mantLo) {
        this.negative = negative;
        this.exponent = exponent;
        this.mantHi = mantHi;
        this.mantLo = mantLo;
    }

    public Quadruple(int exponent, long mantHi, long mantLo) {
        this(false, exponent, mantHi, mantLo);
    }

    public static Quadruple from(String strValue) {
        return new Quadruple(strValue);
    }

    public static Quadruple from(long lValue) {
        return new Quadruple(lValue);
    }

    public static Quadruple from(double dValue) {
        return new Quadruple(dValue);
    }

    public static Quadruple nan() {
        return new Quadruple().assignNaN();
    }

    public static Quadruple zero() {
        return new Quadruple();
    }

    public static Quadruple one() {
        return new Quadruple().assign(1L);
    }

    public static Quadruple two() {
        return new Quadruple().assign(2L);
    }

    public static Quadruple pi() {
        return new Quadruple(Integer.MIN_VALUE, -7917410315110611863L, -8535230491170162270L);
    }

    public int exponent() {
        return this.exponent;
    }

    public long mantHi() {
        return this.mantHi;
    }

    public long mantLo() {
        return this.mantLo;
    }

    public boolean isNegative() {
        return this.negative;
    }

    public boolean isInfinite() {
        return this.exponent == -1 && (this.mantHi | this.mantLo) == 0L;
    }

    public boolean isNaN() {
        return this.exponent == -1 && (this.mantHi | this.mantLo) != 0L;
    }

    public boolean isZero() {
        return (this.mantHi | this.mantLo | (long)this.exponent) == 0L;
    }

    public Quadruple assign(Quadruple qValue) {
        this.negative = qValue.negative;
        this.exponent = qValue.exponent;
        this.mantHi = qValue.mantHi;
        this.mantLo = qValue.mantLo;
        return this;
    }

    public Quadruple assign(double value) {
        long dobleAsLong = Double.doubleToLongBits(value);
        boolean bl = this.negative = (dobleAsLong & Long.MIN_VALUE) != 0L;
        if ((dobleAsLong &= Long.MAX_VALUE) == 0L) {
            return this.assignZero(false);
        }
        this.mantLo = 0L;
        if ((dobleAsLong & 0x7FF0000000000000L) == 0L) {
            return this.makeQuadOfSubnormalDoubleAsLong(dobleAsLong);
        }
        if ((dobleAsLong & 0x7FF0000000000000L) == 0x7FF0000000000000L) {
            this.exponent = -1;
            this.mantHi = (dobleAsLong & 0xFFFFFFFFFFFFFL) == 0L ? 0L : Long.MIN_VALUE;
            return this;
        }
        this.exponent = (int)((dobleAsLong & 0x7FF0000000000000L) >>> 52) - 1023 + Integer.MAX_VALUE;
        this.mantHi = (dobleAsLong & 0xFFFFFFFFFFFFFL) << 12;
        return this;
    }

    public Quadruple assign(long value) {
        if (value == 0L) {
            return this.assignZero();
        }
        if (value == Long.MIN_VALUE) {
            return this.assignWithUnbiasedExponent(true, 63, 0L, 0L);
        }
        if (value < 0L) {
            this.negative = true;
            value = -value;
        } else {
            this.negative = false;
        }
        int bitsToShift = Long.numberOfLeadingZeros(value) + 1;
        value = bitsToShift == 64 ? 0L : value << bitsToShift;
        return this.assignWithUnbiasedExponent(this.negative, 64 - bitsToShift, value, 0L);
    }

    public Quadruple assign(String source) throws NullPointerException, NumberFormatException {
        return QuadrupleParser.parse(source, this);
    }

    public Quadruple assignWithUnbiasedExponent(boolean negative, int exponent, long mantHi, long mantLo) {
        this.negative = negative;
        this.exponent = exponent + Integer.MAX_VALUE;
        this.mantHi = mantHi;
        this.mantLo = mantLo;
        return this;
    }

    public Quadruple assignPositiveInfinity() {
        this.negative = false;
        this.exponent = -1;
        this.mantHi = 0L;
        this.mantLo = 0L;
        return this;
    }

    public Quadruple assignNegativeInfinity() {
        this.negative = true;
        this.exponent = -1;
        this.mantHi = 0L;
        this.mantLo = 0L;
        return this;
    }

    public Quadruple assignNaN() {
        this.negative = false;
        this.exponent = -1;
        this.mantHi = Long.MIN_VALUE;
        this.mantLo = 0L;
        return this;
    }

    @Override
    public int intValue() {
        long exp = ((long)this.exponent & 0xFFFFFFFFL) - Integer.MAX_VALUE;
        if (exp < 0L || this.isNaN()) {
            return 0;
        }
        if (exp >= 31L) {
            return this.negative ? Integer.MIN_VALUE : Integer.MAX_VALUE;
        }
        int intValue = exp == 0L ? 1 : 1 << (int)exp | (int)(this.mantHi >>> (int)(64L - exp));
        return this.negative ? -intValue : intValue;
    }

    @Override
    public long longValue() {
        long exp = ((long)this.exponent & 0xFFFFFFFFL) - Integer.MAX_VALUE;
        if (exp < 0L || this.isNaN()) {
            return 0L;
        }
        if (exp >= 63L) {
            return this.negative ? Long.MIN_VALUE : Long.MAX_VALUE;
        }
        long longValue = exp == 0L ? 1L : 1L << (int)exp | this.mantHi >>> (int)(64L - exp);
        return this.negative ? -longValue : longValue;
    }

    @Override
    public float floatValue() {
        return (float)this.doubleValue();
    }

    @Override
    public double doubleValue() {
        if (this.exponent == 0) {
            return this.negative ? -0.0 : 0.0;
        }
        if (this.exponent == -1) {
            return this.mantHi != 0L || this.mantLo != 0L ? Double.NaN : (this.negative ? Double.NEGATIVE_INFINITY : Double.POSITIVE_INFINITY);
        }
        int expD = this.exponent - Integer.MAX_VALUE;
        if (expD > 1023) {
            return this.negative ? Double.NEGATIVE_INFINITY : Double.POSITIVE_INFINITY;
        }
        if (expD < -1075) {
            return this.negative ? -0.0 : 0.0;
        }
        if (expD < -1022) {
            long lValue = this.mantHi >>> 12 | 0x10000000000000L;
            lValue = lValue + (1L << -1023 - expD) >>> -1023 - expD + 1;
            if (this.negative) {
                lValue |= Long.MIN_VALUE;
            }
            return Double.longBitsToDouble(lValue);
        }
        long dMant = this.mantHi >>> 12;
        if ((this.mantHi & 0x800L) != 0L && ((this.mantHi & 0x7FFL | this.mantLo) != 0L || (dMant & 1L) != 0L) && (++dMant & 0x7FF0000000000000L) != 0L) {
            dMant = (dMant & 0xFFEFFFFFFFFFFFFFL) >>> 1;
            ++expD;
        }
        if (expD > 1023) {
            return this.negative ? Double.NEGATIVE_INFINITY : Double.POSITIVE_INFINITY;
        }
        long lValue = (long)(expD + 1023) << 52 | dMant | (this.negative ? Long.MIN_VALUE : 0L);
        return Double.longBitsToDouble(lValue);
    }

    public String toString() {
        return QuadrupleParser.toString(this);
    }

    @Override
    public int compareTo(Quadruple other) {
        if (this.isNaN()) {
            return other.isNaN() ? 0 : 1;
        }
        if (other.isNaN()) {
            return -1;
        }
        if (this.negative != other.negative) {
            return this.negative ? -1 : 1;
        }
        int result = Integer.compareUnsigned(this.exponent, other.exponent);
        if (result == 0) {
            result = Long.compareUnsigned(this.mantHi, other.mantHi);
        }
        if (result == 0) {
            result = Long.compareUnsigned(this.mantLo, other.mantLo);
        }
        if (this.negative) {
            result = -result;
        }
        return result;
    }

    @Override
    public int compareTo(long other) {
        return this.compareTo(new Quadruple(other));
    }

    @Override
    public int compareTo(double other) {
        return this.compareTo(new Quadruple(other));
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Quadruple)) {
            return false;
        }
        Quadruple other = (Quadruple)obj;
        if (this.isNaN() && other.isNaN()) {
            return false;
        }
        return this.negative == other.negative && this.exponent == other.exponent && this.mantHi == other.mantHi && this.mantLo == other.mantLo;
    }

    public int hashCode() {
        if (this.isNaN()) {
            return -441827835;
        }
        int prime = 31;
        int result = 1;
        result = 31 * result + this.exponent;
        result = 31 * result + Long.hashCode(this.mantHi);
        result = 31 * result + Long.hashCode(this.mantLo);
        result = 31 * result + (this.negative ? 1231 : 1237);
        return result;
    }

    public static int compare(Quadruple q1, Quadruple q2) {
        return q1.compareTo(q2);
    }

    public int compareMagnitudeTo(Quadruple other) {
        if (this.isNaN()) {
            return other.isNaN() ? 0 : 1;
        }
        if (other.isNaN()) {
            return -1;
        }
        if (this.isInfinite()) {
            return other.isInfinite() ? 0 : 1;
        }
        if (other.isInfinite()) {
            return -1;
        }
        int result = Integer.compareUnsigned(this.exponent, other.exponent);
        if (result != 0) {
            return result;
        }
        result = Long.compareUnsigned(this.mantHi, other.mantHi);
        if (result != 0) {
            return result;
        }
        return Long.compareUnsigned(this.mantLo, other.mantLo);
    }

    public static Quadruple max(Quadruple q1, Quadruple q2) {
        if (q1.compareTo(q2) > 0) {
            return new Quadruple(q1);
        }
        return new Quadruple(q2);
    }

    public static Quadruple min(Quadruple q1, Quadruple q2) {
        if (q1.compareTo(q2) < 0) {
            return new Quadruple(q1);
        }
        return new Quadruple(q2);
    }

    public Quadruple add(Quadruple summand) {
        if (this.isNaN() || summand.isNaN()) {
            return this.assignNaN();
        }
        if (this.isInfinite()) {
            if (summand.isInfinite() && this.negative != summand.negative) {
                return this.assignNaN();
            }
            return this;
        }
        if (summand.isInfinite()) {
            return this.assign(summand);
        }
        if (summand.isZero()) {
            if (this.isZero()) {
                this.negative = summand.isNegative() && this.isNegative();
            }
            return this;
        }
        if (this.isZero()) {
            return this.assign(summand);
        }
        if (this.negative == summand.negative) {
            return this.addUnsigned(summand);
        }
        boolean wasNegative = this.negative;
        this.subtractUnsigned(summand);
        this.negative ^= wasNegative;
        return this;
    }

    public Quadruple add(long summand) {
        return this.add(new Quadruple(summand));
    }

    public Quadruple add(double summand) {
        return this.add(new Quadruple(summand));
    }

    public static Quadruple add(Quadruple op1, Quadruple op2) {
        op1 = new Quadruple(op1);
        return op1.add(op2);
    }

    public static Quadruple add(Quadruple op1, long op2) {
        op1 = new Quadruple(op1);
        return op1.add(op2);
    }

    public static Quadruple add(Quadruple op1, double op2) {
        op1 = new Quadruple(op1);
        return op1.add(op2);
    }

    public Quadruple subtract(Quadruple subtrahend) {
        if (this.isNaN() || subtrahend.isNaN()) {
            return this.assignNaN();
        }
        if (this.isInfinite()) {
            if (subtrahend.isInfinite() && this.negative == subtrahend.negative) {
                return this.assignNaN();
            }
            return this;
        }
        if (subtrahend.isInfinite()) {
            return this.assign(subtrahend).negate();
        }
        if (subtrahend.isZero()) {
            if (this.isZero()) {
                this.negative = this.isNegative() && !subtrahend.isNegative();
            }
            return this;
        }
        if (this.isZero()) {
            return this.assign(subtrahend).negate();
        }
        if (this.negative != subtrahend.negative) {
            return this.addUnsigned(subtrahend);
        }
        boolean wasNegative = this.negative;
        this.subtractUnsigned(subtrahend);
        this.negative ^= wasNegative;
        return this;
    }

    public Quadruple subtract(double subtrahend) {
        return this.subtract(new Quadruple(subtrahend));
    }

    public Quadruple multiply(Quadruple factor) {
        if (this.isNaN() || factor.isNaN()) {
            return this.assignNaN();
        }
        if (this.isInfinite()) {
            if (factor.isZero()) {
                return this.assignNaN();
            }
            return this.assignInfinity(factor.negative);
        }
        if (this.isZero()) {
            if (factor.isInfinite()) {
                return this.assignNaN();
            }
            return this.assignZero(factor.negative);
        }
        if (factor.isInfinite()) {
            return this.assignInfinity(factor.negative);
        }
        if (factor.isZero()) {
            return this.assignZero(factor.negative);
        }
        this.multUnsigned(factor);
        this.negative ^= factor.negative;
        return this;
    }

    public Quadruple multiply(long factor) {
        return this.multiply(new Quadruple(factor));
    }

    public Quadruple multiply(double factor) {
        return this.multiply(new Quadruple(factor));
    }

    public Quadruple divide(Quadruple divisor) {
        if (this.isNaN() || divisor.isNaN()) {
            return this.assignNaN();
        }
        if (this.isInfinite()) {
            if (divisor.isInfinite()) {
                return this.assignNaN();
            }
            return this.assignInfinity(divisor.negative);
        }
        if (this.isZero()) {
            if (divisor.isZero()) {
                return this.assignNaN();
            }
            return this.assignZero(divisor.negative);
        }
        if (divisor.isInfinite()) {
            return this.assignZero(divisor.negative);
        }
        if (divisor.isZero()) {
            return this.assignInfinity(divisor.negative);
        }
        this.divideUnsigned(divisor);
        this.negative ^= divisor.negative;
        return this;
    }

    public Quadruple divide(double divisor) {
        return this.divide(new Quadruple(divisor));
    }

    public Quadruple sqrt() {
        if (this.negative) {
            return this.assignNaN();
        }
        if (this.isNaN() || this.isInfinite()) {
            return this;
        }
        long absExp = ((long)this.exponent & 0xFFFFFFFFL) - Integer.MAX_VALUE;
        if (this.exponent == 0) {
            absExp -= this.normalizeMantissa();
        }
        this.exponent = (int)(absExp / 2L + Integer.MAX_VALUE);
        long thirdWord = this.sqrtMant();
        if (absExp % 2L != 0L) {
            long[] multed = this.multBySqrt2(this.mantHi, this.mantLo, thirdWord);
            this.mantHi = multed[0];
            this.mantLo = multed[1];
            thirdWord = multed[2];
            if (absExp < 0L) {
                --this.exponent;
            }
        }
        if ((thirdWord & Long.MIN_VALUE) != 0L && ++this.mantLo == 0L && ++this.mantHi == 0L) {
            ++this.exponent;
        }
        return this;
    }

    public static Quadruple sqrt(Quadruple square) {
        return new Quadruple(square).sqrt();
    }

    public Quadruple setNegative(boolean value) {
        this.negative = value;
        return this;
    }

    public void setExponent(int exponent) {
        this.exponent = exponent;
    }

    public void setMantHi(long mantHi) {
        this.mantHi = mantHi;
    }

    public void setMantLo(long mantLo) {
        this.mantLo = mantLo;
    }

    public Quadruple negate() {
        this.negative = !this.negative;
        return this;
    }

    public Quadruple cpy() {
        return new Quadruple(this);
    }

    public Quadruple abs() {
        return new Quadruple(this.exponent, this.mantHi, this.mantLo);
    }

    public int signum() {
        return this.negative ? -1 : (this.isZero() ? 0 : 1);
    }

    private static void unpack_3x64_to_6x32(long[] qd192, long[] buff_6x32) {
        buff_6x32[0] = qd192[1] >>> 32;
        buff_6x32[1] = qd192[1] & 0xFFFFFFFFL;
        buff_6x32[2] = qd192[2] >>> 32;
        buff_6x32[3] = qd192[2] & 0xFFFFFFFFL;
        buff_6x32[4] = qd192[3] >>> 32;
        buff_6x32[5] = qd192[3] & 0xFFFFFFFFL;
    }

    private static void multPacked3x64_simply() {
        int i;
        long[] buff1232 = Quadruple.buffers.get().BUFFER_12x32;
        long[] buff632a = Quadruple.buffers.get().BUFFER_6x32_A;
        long[] buff632b = Quadruple.buffers.get().BUFFER_6x32_B;
        Arrays.fill(buff1232, 0L);
        Quadruple.unpack_3x64_to_6x32(Quadruple.buffers.get().BUFFER_4x64_A, buff632a);
        Quadruple.unpack_3x64_to_6x32(Quadruple.buffers.get().SQRT_2_AS_LONGS, buff632b);
        for (i = 5; i >= 0; --i) {
            for (int j = 5; j >= 0; --j) {
                long part = buff632a[i] * buff632b[j];
                int n = j + i + 1;
                buff1232[n] = buff1232[n] + (part & 0xFFFFFFFFL);
                int n2 = j + i;
                buff1232[n2] = buff1232[n2] + (part >>> 32);
            }
        }
        i = 11;
        while (i > 0) {
            int n = i - 1;
            buff1232[n] = buff1232[n] + (buff1232[i] >>> 32);
            int n3 = i--;
            buff1232[n3] = buff1232[n3] & 0xFFFFFFFFL;
        }
    }

    private static long[] pack_12x32_to_3x64() {
        long[] buff1232 = Quadruple.buffers.get().BUFFER_12x32;
        long[] buff464 = Quadruple.buffers.get().BUFFER_4x64_A;
        buff464[1] = (buff1232[0] << 32) + buff1232[1];
        buff464[2] = (buff1232[2] << 32) + buff1232[3];
        buff464[3] = (buff1232[4] << 32) + buff1232[5];
        return buff464;
    }

    private static boolean isEmpty(long[] buffer) {
        for (long l : buffer) {
            if (l == 0L) continue;
            return false;
        }
        return true;
    }

    private static String hexStr(long lValue) {
        return String.format("%04x_%04x_%04x_%04x", lValue >> 48 & 0xFFFFL, lValue >> 32 & 0xFFFFL, lValue >> 16 & 0xFFFFL, lValue & 0xFFFFL);
    }

    private static String hexStr(int iValue) {
        return String.format("%04x_%04x", iValue >> 16 & 0xFFFF, iValue & 0xFFFF);
    }

    private Quadruple makeQuadOfSubnormalDoubleAsLong(long doubleAsLong) {
        int numOfZeros = Long.numberOfLeadingZeros(doubleAsLong);
        this.exponent = 2147482624 - (numOfZeros - 12);
        if (numOfZeros < 63) {
            this.mantHi = doubleAsLong << numOfZeros + 1;
        }
        return this;
    }

    public long makeSubnormal(long exp2) {
        if ((exp2 = -exp2) > 127L) {
            this.mantHi = 0L;
            this.mantLo = 0L;
            if (exp2 == 128L) {
                ++this.mantLo;
            }
            return 0L;
        }
        long shiftedOutBit = this.shiftMantissa(exp2);
        exp2 = 0L;
        if (shiftedOutBit != 0L && ++this.mantLo == 0L && ++this.mantHi == 0L) {
            ++exp2;
        }
        return exp2;
    }

    private long shiftMantissa(long exp2) {
        long shiftedOut = this.mantLo & 1L;
        this.mantLo = this.mantLo >>> 1 | this.mantHi << 63;
        this.mantHi = this.mantHi >>> 1 | Long.MIN_VALUE;
        if (exp2 >= 64L) {
            shiftedOut = exp2 == 64L ? this.mantLo >>> 63 : this.mantHi >>> (int)(exp2 - 65L) & 1L;
            this.mantLo = this.mantHi >>> (int)(exp2 - 64L);
            this.mantHi = 0L;
        } else if (exp2 > 0L) {
            shiftedOut = this.mantLo >>> (int)(exp2 - 1L) & 1L;
            this.mantLo = this.mantLo >>> (int)exp2 | this.mantHi << (int)(64L - exp2);
            this.mantHi >>>= (int)exp2;
        }
        return shiftedOut;
    }

    public Quadruple assignMinNormal() {
        this.negative = false;
        this.exponent = 1;
        this.mantHi = 0L;
        this.mantLo = 0L;
        return this;
    }

    public Quadruple assignMaxValue() {
        this.negative = false;
        this.exponent = -2;
        this.mantLo = -1L;
        this.mantHi = -1L;
        return this;
    }

    public Quadruple assignMinValue() {
        this.negative = false;
        this.exponent = 0;
        this.mantHi = 0L;
        this.mantLo = 1L;
        return this;
    }

    public Quadruple assignZero(boolean changeSign) {
        this.negative ^= changeSign;
        this.exponent = 0;
        this.mantHi = this.mantLo = (long)0;
        return this;
    }

    private Quadruple assignZero() {
        this.negative = false;
        this.exponent = 0;
        this.mantHi = this.mantLo = (long)0;
        return this;
    }

    private void assignOne() {
        this.exponent = Integer.MAX_VALUE;
        this.mantHi = 0L;
        this.mantLo = 0L;
    }

    public Quadruple assignInfinity(boolean changeSign) {
        this.negative ^= changeSign;
        this.exponent = -1;
        this.mantHi = 0L;
        this.mantLo = 0L;
        return this;
    }

    private Quadruple addUnsigned(Quadruple summand) {
        if (this.exponent != 0 && summand.exponent != 0) {
            if (this.exponent == summand.exponent) {
                return this.addWithSameExps(summand);
            }
            return this.addWitDifferentExps(summand);
        }
        if ((this.exponent | summand.exponent) != 0) {
            return this.addNormalAndSubnormal(summand);
        }
        this.exponent = (int)this.addMant(summand.mantHi, summand.mantLo);
        return this;
    }

    private Quadruple addWithSameExps(Quadruple summand) {
        long carryUp = this.addMant(summand.mantHi, summand.mantLo);
        long shiftedOutBit = this.mantLo & 1L;
        this.shiftMantissaRight(1);
        if (shiftedOutBit != 0L && ++this.mantLo == 0L) {
            ++this.mantHi;
        }
        if (carryUp != 0L) {
            this.mantHi |= Long.MIN_VALUE;
        }
        if (++this.exponent == -1) {
            this.mantLo = 0L;
            this.mantHi = 0L;
        }
        return this;
    }

    private Quadruple addWitDifferentExps(Quadruple summand) {
        long exp2;
        long greaterLo;
        long greaterHi;
        if (Integer.compareUnsigned(this.exponent, summand.exponent) < 0) {
            greaterHi = summand.mantHi;
            greaterLo = summand.mantLo;
            exp2 = this.exponent;
            this.exponent = summand.exponent;
        } else {
            greaterHi = this.mantHi;
            greaterLo = this.mantLo;
            this.mantHi = summand.mantHi;
            this.mantLo = summand.mantLo;
            exp2 = summand.exponent;
        }
        int shift = this.exponent - (int)exp2;
        if (Integer.compareUnsigned(shift, 129) > 0) {
            this.mantHi = greaterHi;
            this.mantLo = greaterLo;
            return this;
        }
        if (shift == 129) {
            return this.greaterPlusLowerBit(greaterHi, greaterLo);
        }
        long shiftedOutBit = this.shiftAndSetUnity(shift);
        long carryUp = this.addAndRoundUp(greaterHi, greaterLo, shiftedOutBit);
        if (carryUp != 0L) {
            this.shiftAndCorrectExponent(shiftedOutBit);
        }
        return this;
    }

    private Quadruple addNormalAndSubnormal(Quadruple summand) {
        long greaterLo;
        long greaterHi;
        if (this.exponent == 0) {
            greaterHi = summand.mantHi;
            greaterLo = summand.mantLo;
            this.exponent = summand.exponent;
        } else {
            greaterHi = this.mantHi;
            greaterLo = this.mantLo;
            this.mantHi = summand.mantHi;
            this.mantLo = summand.mantLo;
        }
        int shift = this.exponent - 1;
        int lz = Long.numberOfLeadingZeros(this.mantHi);
        if (lz == 64) {
            lz = 64 + Long.numberOfLeadingZeros(this.mantLo);
        }
        if (shift + lz > 128) {
            this.mantHi = greaterHi;
            this.mantLo = greaterLo;
            return this;
        }
        long shiftedOutBit = this.highestShiftedOutBit(shift);
        this.shiftMantissaRight(shift);
        long carryUp = this.addAndRoundUp(greaterHi, greaterLo, shiftedOutBit);
        if (carryUp != 0L) {
            this.shiftAndCorrectExponent(shiftedOutBit);
        }
        return this;
    }

    private long addMant(long mantHi2, long mantLo2) {
        long carry;
        this.mantLo += mantLo2;
        long l = carry = Long.compareUnsigned(this.mantLo, mantLo2) < 0 ? 1L : 0L;
        if (carry != 0L && (this.mantHi += carry) == 0L) {
            this.mantHi = mantHi2;
        } else {
            this.mantHi += mantHi2;
            carry = Long.compareUnsigned(this.mantHi, mantHi2) < 0 ? 1L : 0L;
        }
        return carry;
    }

    private long shiftMantissaRight(int shift) {
        long shiftedOutBits;
        if (shift == 0) {
            return 0L;
        }
        if (shift == 128) {
            long shiftedOutBits2 = this.mantHi;
            this.mantLo = 0L;
            this.mantHi = 0L;
            return shiftedOutBits2;
        }
        long l = shiftedOutBits = shift <= 64 ? this.mantLo << 64 - shift : this.mantHi << 128 - shift | this.mantLo >>> shift - 64;
        if (shift >= 64) {
            this.mantLo = this.mantHi >>> shift - 64;
            this.mantHi = 0L;
        } else {
            this.mantLo = this.mantLo >>> shift | this.mantHi << 64 - shift;
            this.mantHi >>>= shift;
        }
        return shiftedOutBits;
    }

    private Quadruple greaterPlusLowerBit(long greaterHi, long greaterLo) {
        this.mantLo = ++greaterLo;
        if (greaterLo == 0L) {
            this.mantHi = ++greaterHi;
            if (greaterHi == 0L) {
                ++this.exponent;
            }
        } else {
            this.mantHi = greaterHi;
        }
        return this;
    }

    private long shiftAndSetUnity(int shift) {
        long shiftedOutBit = shift == 0 ? 0L : (shift <= 64 ? 1L & this.mantLo >>> shift - 1 : 1L & this.mantHi >>> shift - 65);
        this.shiftMantissaRight(shift);
        if (shift > 64) {
            this.mantLo |= 1L << 128 - shift;
        } else {
            this.mantHi |= 1L << 64 - shift;
        }
        return shiftedOutBit;
    }

    private long addAndRoundUp(long summandHi, long summandLo, long carry) {
        if (carry != 0L && (this.mantLo += carry) == 0L) {
            this.mantLo = summandLo;
            carry = 1L;
        } else {
            this.mantLo += summandLo;
            long l = carry = Long.compareUnsigned(this.mantLo, summandLo) < 0 ? 1L : 0L;
        }
        if (carry != 0L && (this.mantHi += carry) == 0L) {
            this.mantHi = summandHi;
        } else {
            this.mantHi += summandHi;
            carry = Long.compareUnsigned(this.mantHi, summandHi) < 0 ? 1L : 0L;
        }
        return carry;
    }

    private void shiftAndCorrectExponent(long dontRoundUpAnyMore) {
        long shiftedOutBit = dontRoundUpAnyMore != 0L ? 0L : this.mantLo & 1L;
        this.shiftMantissaRight(1);
        if (shiftedOutBit != 0L) {
            this.addMant(0L, 1L);
        }
        if (++this.exponent == -1) {
            this.mantLo = 0L;
            this.mantHi = 0L;
        }
    }

    private long highestShiftedOutBit(int shift) {
        if (shift == 0) {
            return 0L;
        }
        if (shift <= 64) {
            return 1L & this.mantLo >>> shift - 1;
        }
        return 1L & this.mantHi >>> shift - 65;
    }

    private void subtractUnsigned(Quadruple subtrahend) {
        int lesserExp;
        long minuendHi;
        long minuendLo;
        int thisIsGreater = this.compareMagnitudeTo(subtrahend);
        if (thisIsGreater == 0) {
            this.assignZero(false);
            return;
        }
        if (thisIsGreater > 0) {
            minuendLo = this.mantLo;
            minuendHi = this.mantHi;
            this.mantHi = subtrahend.mantHi;
            this.mantLo = subtrahend.mantLo;
            lesserExp = subtrahend.exponent;
            this.negative = false;
        } else {
            minuendLo = subtrahend.mantLo;
            minuendHi = subtrahend.mantHi;
            lesserExp = this.exponent;
            this.exponent = subtrahend.exponent;
            this.negative = true;
        }
        if (this.exponent != 0 && lesserExp != 0) {
            this.subtractNormals(minuendLo, minuendHi, lesserExp);
            return;
        }
        if ((this.exponent | lesserExp) == 0) {
            this.subtractSubnormals(minuendLo, minuendHi);
            return;
        }
        this.subtractSubnormalFromNormal(minuendLo, minuendHi);
    }

    private void subtractNormals(long minuendLo, long minuendHi, int lesserExp) {
        int shift = this.exponent - lesserExp;
        if (shift > 130) {
            this.mantHi = minuendHi;
            this.mantLo = minuendLo;
            return;
        }
        if (shift == 130) {
            this.subtract_1e_130(minuendLo, minuendHi);
            return;
        }
        if (shift == 129) {
            this.subtract_1e_129(minuendLo, minuendHi);
            return;
        }
        if (shift != 0) {
            this.subtractDifferentExp(minuendLo, minuendHi, shift);
            return;
        }
        this.subtractSameExp(minuendLo, minuendHi);
    }

    private void subtract_1e_130(long minuendLo, long minuendHi) {
        if ((this.mantHi | this.mantLo) != 0L && (minuendLo | minuendHi) == 0L) {
            this.mantLo = -1L;
            this.mantHi = -1L;
            --this.exponent;
        } else {
            this.mantHi = minuendHi;
            this.mantLo = minuendLo;
        }
    }

    private void subtract_1e_129(long minuendLo, long minuendHi) {
        long subtrahendHi = this.mantHi;
        long subtrahendLo = this.mantLo;
        if ((minuendHi | minuendLo) == 0L) {
            this.mantLo = -1L;
            this.mantHi = -1L;
            if ((subtrahendHi & Long.MIN_VALUE) != 0L && (subtrahendHi & Long.MAX_VALUE | subtrahendLo) != 0L) {
                --this.mantLo;
            }
            --this.exponent;
        } else {
            this.mantLo = minuendLo;
            this.mantHi = minuendHi;
            if ((subtrahendHi | subtrahendLo) != 0L && --this.mantLo == -1L) {
                --this.mantHi;
            }
        }
    }

    private void subtractDifferentExp(long minuendLo, long minuendHi, int shift) {
        long shiftedOutBits = this.shiftMantissaRight(shift);
        this.setUnity(shift);
        long borrow = Long.compareUnsigned(shiftedOutBits, Long.MIN_VALUE) > 0 ? 1L : 0L;
        borrow = this.subtrMant(minuendHi, minuendLo, borrow);
        if (borrow != 0L) {
            if (shift == 1) {
                this.normalizeShiftedByOneBit(shiftedOutBits);
            } else {
                this.normalizeShiftedByAFewBits(shiftedOutBits);
            }
        } else if ((this.mantHi | this.mantLo) == 0L && shiftedOutBits == Long.MIN_VALUE) {
            --this.exponent;
            this.mantLo = -1L;
            this.mantHi = -1L;
        }
    }

    private void normalizeShiftedByOneBit(long shiftedOutBits) {
        int lz = this.numberOfLeadingZeros();
        if (lz == 0) {
            this.shiftMantLeft(1);
            --this.exponent;
            if (shiftedOutBits != 0L && --this.mantLo == -1L && --this.mantHi == -1L && --this.exponent != 0) {
                this.shiftMantLeft(1);
            }
        } else {
            if (shiftedOutBits != 0L) {
                this.shiftMantLeft(1);
                --this.exponent;
                if (--this.mantLo == -1L) {
                    --this.mantHi;
                }
                lz = this.numberOfLeadingZeros();
            }
            this.normalize(lz + 1);
        }
    }

    private void normalizeShiftedByAFewBits(long shiftedOutBits) {
        this.shiftMantLeft(1);
        --this.exponent;
        if (shiftedOutBits == Long.MIN_VALUE || shiftedOutBits > 0x4000000000000000L) {
            if (--this.mantLo == -1L) {
                --this.mantHi;
            }
        } else if (shiftedOutBits <= -4611686018427387904L) {
            this.mantLo |= 1L;
        }
    }

    private void normalize(int shift) {
        if (Integer.compareUnsigned(this.exponent, shift) > 0) {
            this.shiftMantLeft(shift);
            this.exponent -= shift;
        } else {
            if (this.exponent > 1) {
                this.shiftMantLeft(this.exponent - 1);
            }
            this.exponent = 0;
        }
    }

    private int numberOfLeadingZeros() {
        int lz = Long.numberOfLeadingZeros(this.mantHi);
        if (lz == 64) {
            lz += Long.numberOfLeadingZeros(this.mantLo);
        }
        return lz;
    }

    private void subtractSameExp(long minuendLo, long minuendHi) {
        this.mantLo = minuendLo - this.mantLo;
        if (Long.compareUnsigned(minuendLo, this.mantLo) < 0) {
            --minuendHi;
        }
        this.mantHi = minuendHi - this.mantHi;
        this.normalize(this.numberOfLeadingZeros() + 1);
    }

    private void subtractSubnormalFromNormal(long minuendLo, long minuendHi) {
        int shift = this.exponent - 1;
        int lz = this.numberOfLeadingZeros();
        if ((shift & 0xFFFFFF00) != 0 || shift + lz > 129) {
            this.mantHi = minuendHi;
            this.mantLo = minuendLo;
            return;
        }
        long shiftedOutBits = this.shiftMantissaRight(shift);
        long borrow = Long.compareUnsigned(shiftedOutBits, Long.MIN_VALUE) > 0 ? 1L : 0L;
        if ((borrow = this.subtrMant(minuendHi, minuendLo, borrow)) != 0L) {
            if (shift == 1) {
                this.normalizeShiftedByOneBit(shiftedOutBits);
            } else if (shift != 0) {
                this.normalizeShiftedByAFewBits(shiftedOutBits);
            } else {
                this.exponent = 0;
            }
        } else if ((this.mantHi | this.mantLo) == 0L && (shiftedOutBits == Long.MIN_VALUE || shiftedOutBits > 0x4000000000000000L)) {
            --this.exponent;
            this.mantLo = -1L;
            this.mantHi = -1L;
        }
    }

    private void subtractSubnormals(long minuendLo, long minuendHi) {
        this.mantLo = minuendLo - this.mantLo;
        if (Long.compareUnsigned(this.mantLo, minuendLo) > 0) {
            --minuendHi;
        }
        this.mantHi = minuendHi - this.mantHi;
    }

    private void setUnity(int shift) {
        if (shift > 64) {
            this.mantLo |= 1L << 128 - shift;
        } else {
            this.mantHi |= 1L << 64 - shift;
        }
    }

    private void shiftMantLeft(int shift) {
        assert (shift >= 0 && shift < 129) : "Can't shift by more than 128 or less than 1 bits";
        if (shift == 0) {
            return;
        }
        if (shift == 128) {
            this.mantLo = 0L;
            this.mantHi = 0L;
            return;
        }
        if (shift >= 64) {
            this.mantHi = this.mantLo << shift - 64;
            this.mantLo = 0L;
        } else {
            this.mantHi = this.mantHi << shift | this.mantLo >>> 64 - shift;
            this.mantLo <<= shift;
        }
    }

    private long subtrMant(long minuendHi, long minuendLo, long borrow) {
        if (borrow != 0L && --minuendLo == -1L) {
            this.mantLo = -1L - this.mantLo;
            borrow = 1L;
        } else {
            this.mantLo = minuendLo - this.mantLo;
            long l = borrow = Long.compareUnsigned(this.mantLo, minuendLo) > 0 ? 1L : 0L;
        }
        if (borrow != 0L && --minuendHi == -1L) {
            this.mantHi = -1L - this.mantHi;
        } else {
            this.mantHi = minuendHi - this.mantHi;
            borrow = Long.compareUnsigned(this.mantHi, minuendHi) > 0 ? 1L : 0L;
        }
        return borrow;
    }

    private void multUnsigned(Quadruple factor) {
        long[] factor1 = Quadruple.buffers.get().BUFFER_5x32_A;
        long[] factor2 = Quadruple.buffers.get().BUFFER_5x32_B;
        long[] product = Quadruple.buffers.get().BUFFER_10x32_A;
        long productExponent = Integer.toUnsignedLong(this.exponent) + Integer.toUnsignedLong(factor.exponent) - Integer.MAX_VALUE;
        if (this.exponentWouldExceedBounds(productExponent, 1L, 0L)) {
            return;
        }
        if ((productExponent = this.normalizeAndUnpack(factor, productExponent, factor1, factor2)) < -129L) {
            this.assignZero();
            return;
        }
        Quadruple.multiplyBuffers(factor1, factor2, product);
        boolean isRoundedUp = this.roundBuffer(product);
        productExponent = this.normalizeProduct(product, productExponent, isRoundedUp);
        if (productExponent > 0xFFFFFFFEL) {
            this.assignInfinity(false);
            return;
        }
        this.packBufferToMantissa(product);
        if (productExponent <= 0L) {
            productExponent = this.normalizeSubnormal(productExponent, isRoundedUp);
        }
        this.exponent = (int)productExponent;
    }

    private long normalizeAndUnpack(Quadruple factor, long productExponent, long[] buffer1, long[] buffer2) {
        long factorMantHi = factor.mantHi;
        long factorMantLo = factor.mantLo;
        boolean oneIsSubnormal = false;
        if (this.exponent == 0) {
            oneIsSubnormal = true;
        } else if (factor.exponent == 0) {
            factorMantHi = this.mantHi;
            factorMantLo = this.mantLo;
            this.mantHi = factor.mantHi;
            this.mantLo = factor.mantLo;
            oneIsSubnormal = true;
        }
        if (oneIsSubnormal) {
            int lz = this.numberOfLeadingZeros();
            if ((productExponent -= (long)lz) < -129L) {
                return productExponent;
            }
            this.shiftMantLeft(lz + 1);
        }
        Quadruple.unpack_To5x32(this.mantHi, this.mantLo, buffer1);
        Quadruple.unpack_To5x32(factorMantHi, factorMantLo, buffer2);
        return productExponent;
    }

    private static void multiplyBuffers(long[] factor1, long[] factor2, long[] product) {
        int maxIdxFact;
        assert (factor1.length == factor2.length && product.length == factor1.length * 2) : "Factors' lengths must be equal to each other and twice less than the product's length";
        Arrays.fill(product, 0L);
        for (int i = maxIdxFact = factor1.length - 1; i >= 0; --i) {
            for (int j = maxIdxFact; j >= 0; --j) {
                long sum = factor1[i] * factor2[j];
                int n = i + j + 1;
                product[n] = product[n] + (sum & 0xFFFFFFFFL);
                int n2 = i + j;
                product[n2] = product[n2] + ((sum >>> 32) + (product[i + j + 1] >>> 32));
                int n3 = i + j + 1;
                product[n3] = product[n3] & 0xFFFFFFFFL;
            }
        }
    }

    private boolean roundBuffer(long[] buffer) {
        buffer[6] = buffer[6] + 0x80000000L;
        if ((buffer[6] & 0x100000000L) == 0L) {
            return false;
        }
        int i = 5;
        while (true) {
            buffer[i + 1] = 0L;
            int n = i;
            buffer[n] = buffer[n] + 1L;
            if ((buffer[i] & 0x100000000L) == 0L) break;
            --i;
        }
        return true;
    }

    private long normalizeProduct(long[] product, long productExponent, boolean isRoundedUp) {
        if (product[1] > 1L && ++productExponent <= 0xFFFFFFFEL) {
            this.shiftBufferRight(product, isRoundedUp);
        }
        return productExponent;
    }

    private void packBufferToMantissa(long[] buffer) {
        this.mantLo = buffer[5] & 0xFFFFFFFFL | buffer[4] << 32;
        this.mantHi = buffer[3] & 0xFFFFFFFFL | buffer[2] << 32;
    }

    private long normalizeSubnormal(long productExponent, boolean isRoundedUp) {
        if (isRoundedUp) {
            this.mantLo &= 0xFFFFFFFFFFFFFFFEL;
        }
        productExponent = this.makeSubnormal(productExponent);
        return productExponent;
    }

    private void shiftBufferRight(long[] buffer, boolean isRoundedUp) {
        if (isRoundedUp) {
            this.shiftBuffRightWithoutRounding(buffer);
        } else {
            this.shiftBuffRightWithRounding(buffer);
        }
    }

    private void shiftBuffRightWithRounding(long[] buffer) {
        long carry = buffer[5] & 1L;
        this.shiftBuffRightWithoutRounding(buffer);
        buffer[5] = buffer[5] + carry;
        for (int i = 5; i >= 2; --i) {
            if ((buffer[i] & 0xFFFFFFFF00000000L) == 0L) continue;
            int n = i;
            buffer[n] = buffer[n] & 0xFFFFFFFFL;
            int n2 = i - 1;
            buffer[n2] = buffer[n2] + 1L;
        }
    }

    private void shiftBuffRightWithoutRounding(long[] buffer) {
        for (int i = 5; i >= 2; --i) {
            buffer[i] = buffer[i] >>> 1 | (buffer[i - 1] & 1L) << 31;
        }
    }

    private static void unpack_To5x32(long mantHi, long mantLo, long[] buffer) {
        buffer[0] = 1L;
        buffer[1] = mantHi >>> 32;
        buffer[2] = mantHi & 0xFFFFFFFFL;
        buffer[3] = mantLo >>> 32;
        buffer[4] = mantLo & 0xFFFFFFFFL;
    }

    private static void unpack_To5x32(long mantHi, long mantLo, int[] buffer) {
        buffer[0] = 1;
        buffer[1] = (int)(mantHi >>> 32);
        buffer[2] = (int)mantHi;
        buffer[3] = (int)(mantLo >>> 32);
        buffer[4] = (int)mantLo;
    }

    private void divideUnsigned(Quadruple divisor) {
        if (divisor.compareMagnitudeTo(ONE) == 0) {
            return;
        }
        if (this.compareMagnitudeTo(divisor) == 0) {
            this.assignOne();
            return;
        }
        long quotientExponent = Integer.toUnsignedLong(this.exponent) - Integer.toUnsignedLong(divisor.exponent) + Integer.MAX_VALUE;
        if (this.exponentWouldExceedBounds(quotientExponent, 0L, 1L)) {
            return;
        }
        boolean needToDivide = true;
        int[] divisorBuff = Quadruple.buffers.get().BUFFER_5x32_A_INT;
        if (this.exponent != 0 & divisor.exponent != 0) {
            if (this.mantHi == divisor.mantHi && this.mantLo == divisor.mantLo) {
                this.mantLo = 0L;
                this.mantHi = 0L;
                needToDivide = false;
            } else if (divisor.mantHi == 0L && divisor.mantLo == 0L) {
                needToDivide = false;
            } else {
                Quadruple.unpack_To5x32(divisor.mantHi, divisor.mantLo, divisorBuff);
            }
        } else {
            quotientExponent = this.normalizeAndUnpackSubnormals(quotientExponent, divisor, divisorBuff);
        }
        if (needToDivide) {
            quotientExponent = this.doDivide(quotientExponent, divisorBuff);
        }
        if (this.exponentWouldExceedBounds(quotientExponent, 0L, 0L)) {
            return;
        }
        if (quotientExponent <= 0L) {
            quotientExponent = this.makeSubnormal(quotientExponent);
        }
        this.exponent = (int)quotientExponent;
    }

    private boolean exponentWouldExceedBounds(long exponent, long lowerBoundTolerance, long upperBoundTolerance) {
        if (exponent > 0xFFFFFFFEL + upperBoundTolerance) {
            this.assignInfinity(false);
            return true;
        }
        if (exponent < -(128L + lowerBoundTolerance)) {
            this.assignZero(false);
            return true;
        }
        return false;
    }

    private long normalizeAndUnpackSubnormals(long quotientExponent, Quadruple divisor, int[] divisorBuff) {
        if (this.exponent == 0) {
            quotientExponent -= this.normalizeMantissa();
        }
        if (divisor.exponent == 0) {
            quotientExponent += Quadruple.normalizeAndUnpackDivisor(divisor, divisorBuff);
        } else {
            Quadruple.unpack_To5x32(divisor.mantHi, divisor.mantLo, divisorBuff);
        }
        return quotientExponent;
    }

    private long normalizeMantissa() {
        int shift = Long.numberOfLeadingZeros(this.mantHi);
        if (shift == 64) {
            shift += Long.numberOfLeadingZeros(this.mantLo);
        }
        this.shiftMantLeft(shift + 1);
        return shift;
    }

    private static long normalizeAndUnpackDivisor(Quadruple divisor, int[] buffer) {
        long mantHi = divisor.mantHi;
        long mantLo = divisor.mantLo;
        int shift = Long.numberOfLeadingZeros(mantHi);
        if (shift == 64) {
            shift += Long.numberOfLeadingZeros(mantLo);
        }
        long result = shift;
        if (++shift <= 64) {
            mantHi = (mantHi << shift) + (mantLo >>> 64 - shift);
            mantLo <<= shift;
        } else {
            mantHi = mantLo << shift - 64;
            mantLo = 0L;
        }
        Quadruple.unpack_To5x32(mantHi, mantLo, buffer);
        return result;
    }

    private long doDivide(long quotientExponent, int[] divisor) {
        int[] dividend = Quadruple.buffers.get().BUFFER_10x32_A_INT;
        quotientExponent = this.unpackMantissaTo(quotientExponent, divisor, dividend);
        this.divideBuffers(dividend, divisor, quotientExponent);
        return quotientExponent;
    }

    private long unpackMantissaTo(long quotientExponent, int[] divisor, int[] dividend) {
        if (this.compareMantissaWith(divisor) < 0) {
            this.unpackDoubledMantissaToBuff_10x32(dividend);
            --quotientExponent;
        } else {
            this.unpackMantissaToBuff_10x32(dividend);
        }
        return quotientExponent;
    }

    private int compareMantissaWith(int[] divisor) {
        int cmp = Long.compareUnsigned(this.mantHi, (long)divisor[1] << 32 | (long)divisor[2] & 0xFFFFFFFFL);
        return cmp == 0 ? Long.compareUnsigned(this.mantLo, (long)divisor[3] << 32 | (long)divisor[4] & 0xFFFFFFFFL) : cmp;
    }

    private void unpackDoubledMantissaToBuff_10x32(int[] buffer) {
        buffer[0] = 0;
        buffer[1] = (int)(2L + (this.mantHi >>> 63));
        buffer[2] = (int)(this.mantHi >>> 31 & 0xFFFFFFFFL);
        buffer[3] = (int)((this.mantHi << 1) + (this.mantLo >>> 63) & 0xFFFFFFFFL);
        buffer[4] = (int)(this.mantLo >>> 31 & 0xFFFFFFFFL);
        buffer[5] = (int)(this.mantLo << 1 & 0xFFFFFFFFL);
        for (int i = 6; i < 10; ++i) {
            buffer[i] = 0;
        }
    }

    private void unpackMantissaToBuff_10x32(int[] buffer) {
        buffer[0] = 0;
        buffer[1] = 1;
        buffer[2] = (int)(this.mantHi >>> 32);
        buffer[3] = (int)(this.mantHi & 0xFFFFFFFFL);
        buffer[4] = (int)(this.mantLo >>> 32);
        buffer[5] = (int)(this.mantLo & 0xFFFFFFFFL);
        for (int i = 6; i < 10; ++i) {
            buffer[i] = 0;
        }
    }

    private void divideBuffers(int[] dividend, int[] divisor, long quotientExponent) {
        int[] quotientBuff = Quadruple.buffers.get().BUFFER_5x32_B_INT;
        long nextBit = Quadruple.divideArrays(dividend, divisor, quotientBuff);
        this.packMantissaFromWords_1to4(quotientBuff);
        if (quotientExponent > 0L && nextBit != 0L && ++this.mantLo == 0L) {
            ++this.mantHi;
        }
    }

    private static long divideArrays(int[] dividend, int[] divisor, int[] quotient) {
        long divisorHigh = (long)divisor[0] << 32 | (long)divisor[1] & 0xFFFFFFFFL;
        int offset = 0;
        quotient[offset++] = 1;
        Quadruple.subtractDivisor(divisor, dividend);
        do {
            long quotientWord;
            long remainderHigh = (long)dividend[offset + 1] << 32 | (long)dividend[offset + 2] & 0xFFFFFFFFL;
            long l = quotientWord = dividend[offset] == 0 ? Quadruple.divideUnsignedLongs(remainderHigh, divisorHigh) : Quadruple.divide65bits(dividend[offset], remainderHigh, divisorHigh);
            if (quotientWord == 0x100000000L) {
                --quotientWord;
            }
            if (quotientWord != 0L && Quadruple.multiplyAndSubtract(divisor, quotientWord, offset, dividend) < 0) {
                --quotientWord;
                Quadruple.addDivisorBack(divisor, dividend, offset);
            }
            quotient[offset++] = (int)quotientWord;
        } while (offset <= 4);
        if (Quadruple.greaterThanHalfOfDivisor_3(dividend, divisor, offset)) {
            return 1L;
        }
        return 0L;
    }

    private static void subtractDivisor(int[] divisor, int[] remainder) {
        long carry = 0L;
        for (int i = 5; i >= 1; --i) {
            long difference = ((long)remainder[i] & 0xFFFFFFFFL) - ((long)divisor[i - 1] & 0xFFFFFFFFL) + carry;
            remainder[i] = (int)difference;
            carry = difference >> 32;
        }
    }

    private static long divideUnsignedLongs(long dividend, long divisor) {
        long dividendHi = dividend >>> 16;
        long remainder = dividendHi % divisor << 16 | dividend & 0xFFFFL;
        return dividendHi / divisor << 16 | remainder / divisor;
    }

    private static long divide65bits(long dividendHi, long dividendLo, long divisor) {
        dividendHi = dividendHi << 48 | dividendLo >>> 16;
        long quotientHi = dividendHi / divisor;
        long remainder = dividendHi % divisor << 16 | dividendLo & 0xFFFFL;
        long quotientLo = remainder / divisor;
        return quotientHi << 16 | quotientLo;
    }

    private static int multiplyAndSubtract(int[] divisor, long quotientWord, int offset, int[] remainder) {
        offset += 5;
        long carry = 0L;
        long difference = 0L;
        for (int i = 4; i >= 0; --i) {
            long product = quotientWord * ((long)divisor[i] & 0xFFFFFFFFL) + carry;
            difference = (long)remainder[offset] - product;
            remainder[offset--] = (int)difference;
            carry = product >>> 32;
            if ((difference & 0xFFFFFFFFL) <= ((long)(~((int)product)) & 0xFFFFFFFFL)) continue;
            ++carry;
        }
        return (int)difference;
    }

    private static void addDivisorBack(int[] divisor, int[] remainder, int offset) {
        offset += 5;
        long carry = 0L;
        for (int idx = 4; idx >= 0; --idx) {
            long sum = ((long)remainder[offset] & 0xFFFFFFFFL) + ((long)divisor[idx] & 0xFFFFFFFFL) + carry;
            remainder[offset--] = (int)sum;
            carry = sum >>> 32;
        }
    }

    private static boolean greaterThanHalfOfDivisor_3(int[] remainder, int[] divisor, int offset) {
        for (int idx = 0; idx < 4; ++idx) {
            int cmp;
            if ((cmp = Integer.compare((remainder[offset] << 1) + (remainder[++offset] >>> 31) + Integer.MIN_VALUE, divisor[idx] + Integer.MIN_VALUE)) > 0) {
                return true;
            }
            if (cmp >= 0) continue;
            return false;
        }
        int cmp = Integer.compareUnsigned(remainder[offset] << 1, divisor[4]);
        return cmp >= 0;
    }

    private void packMantissaFromWords_1to4(int[] buffer) {
        this.mantLo = (long)buffer[4] & 0xFFFFFFFFL | (long)buffer[3] << 32;
        this.mantHi = (long)buffer[2] & 0xFFFFFFFFL | (long)buffer[1] << 32;
    }

    private long sqrtMant() {
        long[] remainder = Quadruple.buffers.get().BUFFER_3x64_A;
        long[] rootX2 = Quadruple.buffers.get().BUFFER_3x64_B;
        Arrays.fill(rootX2, 0L);
        long[] root = Quadruple.buffers.get().BUFFER_3x64_C;
        Arrays.fill(root, 0L);
        long digit = this.findFirstDigit();
        remainder[0] = this.mantHi - ((512L + digit) * digit << 48);
        remainder[1] = this.mantLo;
        remainder[2] = 0L;
        Quadruple.shift_6_bitsLeft(remainder);
        root[0] = digit << 56;
        rootX2[0] = 0x80000000000000L + (digit << 47);
        int bitNumber = 8;
        if (!Quadruple.isEmpty(remainder)) {
            while (bitNumber < 160) {
                bitNumber = Quadruple.computeNextDigit(remainder, rootX2, root, bitNumber);
            }
        }
        this.mantHi = root[0];
        this.mantLo = root[1];
        return root[2];
    }

    private long findFirstDigit() {
        int sqrtDigit = (int)(this.mantHi >>> 48);
        int idx = Arrays.binarySearch(SQUARE_BYTES, sqrtDigit);
        if (idx < 0) {
            idx = -idx - 2;
        }
        return ROOT_BYTES[idx];
    }

    private static void shift_6_bitsLeft(long[] buffer) {
        for (int i = 0; i < buffer.length - 1; ++i) {
            buffer[i] = buffer[i] << 6 | buffer[i + 1] >>> 58;
        }
        int n = buffer.length - 1;
        buffer[n] = buffer[n] << 6;
    }

    private static int computeNextDigit(long[] remainder, long[] rootX2, long[] root, int bitNumber) {
        long[] aux = Quadruple.buffers.get().BUFFER_3x64_D;
        long digit = Quadruple.findNextDigit(rootX2, remainder, aux, bitNumber);
        Quadruple.addDigit(root, digit, bitNumber);
        boolean remainderIsEmpty = Quadruple.subtractBuff(aux, remainder);
        if (remainderIsEmpty || bitNumber >= 152) {
            return Integer.MAX_VALUE;
        }
        Quadruple.shift_8_bitsLeft(remainder);
        Quadruple.addDigitToBuff(rootX2, digit, bitNumber + 9);
        return bitNumber += 8;
    }

    private static long findNextDigit(long[] rootX2, long[] remainder, long[] aux, int rootBitNumber) {
        long digit = Long.divideUnsigned(remainder[0], rootX2[0]);
        digit = Math.min(digit, 255L);
        Quadruple.computeAux(digit, rootBitNumber, rootX2, aux);
        while (Quadruple.compareBuffs64(aux, remainder) > 0) {
            Quadruple.computeAux(--digit, rootBitNumber, rootX2, aux);
        }
        return digit;
    }

    private static void addDigit(long[] root, long digit, int rootBitNumber) {
        int buffIdx = rootBitNumber / 64;
        int bitIdx = rootBitNumber % 64;
        int n = buffIdx;
        root[n] = root[n] + (digit << 56 - bitIdx);
    }

    private static boolean subtractBuff(long[] subtrahend, long[] minuend) {
        boolean diffIsEmpty = true;
        for (int i = subtrahend.length - 1; i >= 0; --i) {
            long minnd = minuend[i];
            long diff = minnd - subtrahend[i];
            if (Long.compareUnsigned(diff, minnd) > 0) {
                if (minuend[i - 1] == 0L) {
                    int n = i - 1;
                    subtrahend[n] = subtrahend[n] + 1L;
                } else {
                    int n = i - 1;
                    minuend[n] = minuend[n] - 1L;
                }
            }
            minuend[i] = diff;
            diffIsEmpty &= diff == 0L;
        }
        return diffIsEmpty;
    }

    private static void shift_8_bitsLeft(long[] buffer) {
        for (int i = 0; i < buffer.length - 1; ++i) {
            buffer[i] = buffer[i] << 8 | buffer[i + 1] >>> 56;
        }
        int n = buffer.length - 1;
        buffer[n] = buffer[n] << 8;
    }

    private static void addDigitToBuff(long[] buff, long digit, int bitNumber) {
        int buffIdx = bitNumber / 64;
        int bitIdx = bitNumber % 64;
        if (bitIdx <= 56) {
            int n = buffIdx;
            buff[n] = buff[n] + (digit << 56 - bitIdx);
        } else {
            int n = buffIdx;
            buff[n] = buff[n] + (digit >>> bitIdx + 8 - 64);
            int n2 = buffIdx + 1;
            buff[n2] = buff[n2] + (digit << 120 - bitIdx);
        }
    }

    private static void computeAux(long digit, int rootBitNumber, long[] rootX2, long[] aux) {
        Quadruple.copyBuff(rootX2, aux);
        Quadruple.addDigitToBuff(aux, digit, rootBitNumber + 10);
        Quadruple.multBufByDigit(aux, digit);
    }

    private static void copyBuff(long[] src, long[] dst) {
        System.arraycopy(src, 0, dst, 0, src.length);
    }

    private static int compareBuffs64(long[] buff1, long[] buff2) {
        for (int i = 0; i < buff1.length; ++i) {
            if (buff1[i] == buff2[i]) continue;
            return Long.compareUnsigned(buff1[i], buff2[i]);
        }
        return 0;
    }

    private static void multBufByDigit(long[] buff, long digit) {
        long carry = 0L;
        for (int i = buff.length - 1; i >= 0; --i) {
            long prodLo = (buff[i] & 0xFFFFFFFFL) * digit + carry;
            long prodHi = (buff[i] >>> 32) * digit;
            carry = prodHi >>> 32;
            long product = prodLo + (prodHi << 32);
            if (Long.compareUnsigned(product, prodHi << 32) < 0) {
                ++carry;
            }
            buff[i] = product;
        }
    }

    private long[] multBySqrt2(long mantHi, long mantLo, long thirdWord) {
        long[] buff464 = Quadruple.buffers.get().BUFFER_4x64_A;
        buff464[0] = 0L;
        buff464[1] = mantHi >>> 1 | Long.MIN_VALUE;
        buff464[2] = mantLo >>> 1 | mantHi << 63;
        buff464[3] = thirdWord >>> 1 | mantLo << 63;
        long[] product = Quadruple.multPacked3x64();
        product[0] = product[1] << 2 | product[2] >>> 62;
        product[1] = product[2] << 2 | product[3] >>> 62;
        product[2] = product[3] << 2;
        return product;
    }

    private static long[] multPacked3x64() {
        Quadruple.multPacked3x64_simply();
        return Quadruple.pack_12x32_to_3x64();
    }

    private static class Buffers {
        final long[] BUFFER_4x64_A = new long[4];
        final long[] BUFFER_3x64_A = new long[3];
        final long[] BUFFER_3x64_B = new long[3];
        final long[] BUFFER_3x64_C = new long[3];
        final long[] BUFFER_3x64_D = new long[3];
        final long[] BUFFER_5x32_A = new long[5];
        final int[] BUFFER_5x32_A_INT = new int[5];
        final long[] BUFFER_5x32_B = new long[5];
        final int[] BUFFER_5x32_B_INT = new int[5];
        final long[] BUFFER_6x32_A = new long[6];
        final long[] BUFFER_6x32_B = new long[6];
        final long[] BUFFER_10x32_A = new long[10];
        final int[] BUFFER_10x32_A_INT = new int[10];
        final long[] BUFFER_12x32 = new long[12];
        final long[] SQRT_2_AS_LONGS = new long[]{0L, -5402926248376769404L, 6448461645324402335L, 2121020303797364813L};

        private Buffers() {
        }
    }
}

